<?php
// T1.php - Enviar datos a la tabla final (solo T1) usando Cestudiante

// ACTIVAR errores para debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json; charset=utf-8');

try {
    // CONEXIÓN DESDE ARCHIVO EXTERNO
    require_once 'config_db.php';
    
    $conexion = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    
    if ($conexion->connect_error) {
        throw new Exception('Error conectando a la base de datos: ' . $conexion->connect_error);
    }
    
    $conexion->set_charset(DB_CHARSET);

    // **CAMBIADO: Ahora recibimos 'cestudiantes' en lugar de 'ids'**
    if (!isset($_POST['cestudiantes']) || empty($_POST['cestudiantes'])) {
        throw new Exception('No se recibieron códigos de estudiante (cestudiantes)');
    }
    
    $cestudiantes = $_POST['cestudiantes'];
    
    // Si viene como JSON string (desde fetch), decodificarlo
    if (is_string($cestudiantes) && $cestudiantes[0] === '[') {
        $cestudiantes = json_decode($cestudiantes, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Error decodificando JSON de cestudiantes: ' . json_last_error_msg());
        }
    }
    
    // **MODIFICADO: Validación flexible para cédulas (acepta letras y números)**
    $cestudiantes_limpios = array_filter($cestudiantes, function($codigo) {
        // Aceptar strings no vacíos (pueden contener números, letras, guiones, etc.)
        if (is_string($codigo) && trim($codigo) !== '') {
            return true;
        }
        // También aceptar números directamente
        if (is_numeric($codigo)) {
            return true;
        }
        return false;
    });
    
    if (empty($cestudiantes_limpios)) {
        // DEBUG: Mostrar lo que recibimos
        error_log("Datos recibidos en POST: " . print_r($_POST, true));
        error_log("Tipo de cestudiantes: " . gettype($cestudiantes));
        if (is_array($cestudiantes)) {
            error_log("Primeros 3 cestudiantes: " . implode(', ', array_slice($cestudiantes, 0, 3)));
        }
        throw new Exception('No hay códigos de estudiante válidos');
    }
    
    // **MODIFICADO: Limpieza más general para cédulas con letras**
    $cestudiantes_limpios = array_map(function($codigo) {
        // Convertir a string
        $codigo_str = strval($codigo);
        // Eliminar espacios al inicio y final
        $codigo_str = trim($codigo_str);
        // Mantener letras, números, guiones, puntos - común en identificaciones
        // Ejemplos: "1723456789", "X-123456", "PAS-123456", "12345678-9"
        $codigo_str = preg_replace('/[^\w\.\-]/', '', $codigo_str);
        return $codigo_str;
    }, $cestudiantes_limpios);
    
    // Filtrar cadenas vacías después de la limpieza
    $cestudiantes_limpios = array_filter($cestudiantes_limpios, function($codigo) {
        return !empty($codigo);
    });
    
    // Ordenar para mejor legibilidad (orden alfabético/numerico mixto)
    sort($cestudiantes_limpios, SORT_STRING);
    
    error_log("=== PROCESANDO " . count($cestudiantes_limpios) . " CESTUDIANTES PARA TABLA FINAL (T1) ===");
    error_log("Cestudiantes: " . implode(', ', $cestudiantes_limpios));
    
    // Verificar longitudes y tipos de cédulas para depuración
    $longitudes = array_map('strlen', $cestudiantes_limpios);
    error_log("Longitudes de cédulas: " . implode(', ', $longitudes));
    
    // Mostrar ejemplos de diferentes formatos
    $formatos_ejemplo = [];
    foreach ($cestudiantes_limpios as $c) {
        if (preg_match('/^[0-9]+$/', $c)) {
            $formatos_ejemplo[] = "numérico: $c";
        } elseif (preg_match('/^[A-Za-z]/', $c)) {
            $formatos_ejemplo[] = "con letras: $c";
        } elseif (preg_match('/[A-Za-z]/', $c)) {
            $formatos_ejemplo[] = "mixto: $c";
        }
    }
    if (!empty($formatos_ejemplo)) {
        error_log("Formatos encontrados: " . implode(', ', array_slice($formatos_ejemplo, 0, 5)));
    }
    
    // Crear placeholders
    $placeholders = str_repeat('?,', count($cestudiantes_limpios) - 1) . '?';
    
    // **MODIFICADO: Agregar Asignatura a la consulta SELECT**
    $query = "SELECT id, Docente, Especialidad, Curso, Paralelo,
                     Cestudiante, Estudiante, Asignatura, `T1` as T1_Calificacion                     
              FROM concentrado WHERE Cestudiante IN ($placeholders)";
    
    error_log("Query SELECT: " . $query);
    
    $stmt = $conexion->prepare($query);
    if (!$stmt) {
        throw new Exception('Error preparando SELECT: ' . $conexion->error);
    }
    
    // **CAMBIADO: 's' para strings (acepta letras y números)**
    $types = str_repeat('s', count($cestudiantes_limpios));
    $stmt->bind_param($types, ...$cestudiantes_limpios);
    
    if (!$stmt->execute()) {
        throw new Exception('Error ejecutando SELECT: ' . $stmt->error);
    }
    
    $result = $stmt->get_result();
    $total_estudiantes = $result->num_rows;
    error_log("Estudiantes encontrados en concentrado: " . $total_estudiantes);
    
    // Verificar si encontramos todos los estudiantes
    if ($total_estudiantes < count($cestudiantes_limpios)) {
        $encontrados = array();
        while ($row = $result->fetch_assoc()) {
            $encontrados[] = $row['Cestudiante'];
        }
        $no_encontrados = array_diff($cestudiantes_limpios, $encontrados);
        
        // Mostrar detalles de los no encontrados
        if (!empty($no_encontrados)) {
            error_log("Cestudiantes no encontrados en concentrado (" . count($no_encontrados) . "):");
            foreach ($no_encontrados as $no_encontrado) {
                error_log("  - '$no_encontrado' (tipo: " . gettype($no_encontrado) . 
                         ", longitud: " . strlen($no_encontrado) . ")");
            }
        }
        
        // Reiniciar el resultado para procesar
        $result->data_seek(0);
    }
    
    // Obtener el año actual para la columna Año
    $anio_actual = date('Y');
    $fecha_actual = date('d/m/Y H:i');
    
    $insertados = 0;
    $actualizados = 0;
    $errores = 0;
    $procesados = [];
    
    while ($estudiante = $result->fetch_assoc()) {
        $id_actual = $estudiante['id'];
        $cestudiante = $estudiante['Cestudiante'];
        $asignatura = isset($estudiante['Asignatura']) ? $estudiante['Asignatura'] : '';
        
        error_log("Procesando Cestudiante: " . $cestudiante . " - " . $estudiante['Estudiante']);
        $procesados[] = $cestudiante;
        
        // Calificación T1
        $t1_calificacion = floatval($estudiante['T1_Calificacion']);
        $t1f = $t1_calificacion; // Para T1, T1F es igual a T1
        $observaciones = "Actualizado T1 - " . $fecha_actual;
        
        // VERIFICAR SI CESTUDIANTE YA EXISTE EN TABLA FINAL
        $check = $conexion->prepare("SELECT id, Cestudiante, T1, T2, T3 FROM `final` WHERE Cestudiante = ?");
        $check->bind_param("s", $cestudiante);
        $check->execute();
        $check_result = $check->get_result();
        
        if ($check_result->num_rows > 0) {
            // ACTUALIZAR REGISTRO EXISTENTE
            $registro_existente = $check_result->fetch_assoc();
            $id_final = $registro_existente['id'];
            $t1_anterior = $registro_existente['T1'];
            
            error_log("✓ Cestudiante '$cestudiante' ya existe (ID: $id_final) - ACTUALIZANDO T1");
            error_log("  T1 anterior: $t1_anterior → T1 nuevo: $t1_calificacion");
            
            // Obtener valores actuales para calcular promedio
            $t2_actual = floatval($registro_existente['T2']);
            $t3_actual = floatval($registro_existente['T3']);
            
            // Calcular nuevo promedio con los 3 trimestres
            // Si T2 y T3 son 0, solo usar T1 para el promedio
            if ($t2_actual == 0 && $t3_actual == 0) {
                $nuevo_promedio = $t1_calificacion;
            } else {
                $nuevo_promedio = ($t1_calificacion + $t2_actual + $t3_actual) / 3;
            }
            
            // Actualizar solo T1, T1F, Promedio, PromedioF y Observaciones
            $update = $conexion->prepare("UPDATE `final` SET 
                T1 = ?, 
                T1F = ?, 
                Promedio = ?, 
                PromedioF = ?, 
                Observaciones = CONCAT(IFNULL(Observaciones, ''), ' | ', ?)
                WHERE Cestudiante = ?");
            
            if (!$update) {
                $error_msg = "Error preparando UPDATE para Cestudiante '$cestudiante': " . $conexion->error;
                error_log($error_msg);
                $errores++;
                continue;
            }
            
            $update->bind_param(
                "ddddss",  // 6 parámetros
                $t1_calificacion,   // d (T1)
                $t1f,               // d (T1F)
                $nuevo_promedio,    // d (Promedio)
                $nuevo_promedio,    // d (PromedioF)
                $observaciones,     // s (Observaciones)
                $cestudiante        // s (Cestudiante)
            );
            
            if ($update->execute()) {
                $filas_afectadas = $update->affected_rows;
                if ($filas_afectadas > 0) {
                    error_log("  ✓ T1 actualizado correctamente para '$cestudiante'");
                    $actualizados++;
                } else {
                    error_log("  ⚠ No se realizaron cambios para '$cestudiante'");
                }
            } else {
                $error_msg = "✗ Error actualizando '$cestudiante': " . $update->error;
                error_log($error_msg);
                $errores++;
            }
            
            $update->close();
            
        } else {
            // INSERTAR NUEVO REGISTRO (Cestudiante no existe)
            error_log("✓ Cestudiante '$cestudiante' no existe - INSERTANDO nuevo registro");
            
            // **MODIFICADO: Agregar Asignatura al INSERT**
            $insert = $conexion->prepare("INSERT INTO `final` (
                Docente, Especialidad, Curso, Paralelo, 
                Cestudiante, Estudiante, Asignatura,
                T1, T1F, T2, T2F, T3, T3F, 
                Promedio, PromedioF, Supletorio, FinalF, Final, 
                Observaciones, Año
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            
            if (!$insert) {
                $error_msg = "Error preparando INSERT para '$cestudiante': " . $conexion->error;
                error_log($error_msg);
                $errores++;
                continue;
            }
            
            // Valores por defecto para campos no utilizados en T1
            $t2 = 0.0;
            $t2f = 0.0;
            $t3 = 0.0;
            $t3f = 0.0;
            $promedio = $t1_calificacion; // Solo T1 por ahora
            $promediof = $t1_calificacion; // Solo T1 por ahora
            $supletorio = 0.0;
            $finalf = 0.0;
            $final_calificacion = 0.0;
            $observaciones_insert = "Cargado desde T1 - " . $fecha_actual;
            
            // **MODIFICADO: Bind parameters - ahora 20 parámetros**
            $insert->bind_param(
                "sssssssdddddddddddss",  // 20 parámetros
                $estudiante['Docente'],         // s
                $estudiante['Especialidad'],    // s
                $estudiante['Curso'],           // s
                $estudiante['Paralelo'],        // s
                $estudiante['Cestudiante'],     // s
                $estudiante['Estudiante'],      // s
                $asignatura,                    // s (Asignatura - NUEVO)
                $t1_calificacion,               // d (T1)
                $t1f,                           // d (T1F)
                $t2,                            // d (T2)
                $t2f,                           // d (T2F)
                $t3,                            // d (T3)
                $t3f,                           // d (T3F)
                $promedio,                      // d (Promedio)
                $promediof,                     // d (PromedioF)
                $supletorio,                    // d (Supletorio)
                $finalf,                        // d (FinalF)
                $final_calificacion,            // d (Final)
                $observaciones_insert,          // s (Observaciones)
                $anio_actual                    // s (Año)
            );
            
            if ($insert->execute()) {
                $nuevo_id = $conexion->insert_id;
                error_log("  ✓ Nuevo registro insertado para '$cestudiante' (ID: $nuevo_id)");
                $insertados++;
            } else {
                $error_msg = "✗ Error insertando '$cestudiante' en tabla final: " . $insert->error;
                error_log($error_msg);
                $errores++;
            }
            
            $insert->close();
        }
    }
    
    // Verificar si hubo Cestudiantes que no se encontraron en concentrado
    $no_encontrados = array_diff($cestudiantes_limpios, $procesados);
    $total_no_encontrados = count($no_encontrados);
    
    // Respuesta final
    $mensaje = "Proceso completado. ";
    
    if ($insertados > 0) {
        $mensaje .= "$insertados nuevos registros insertados. ";
    }
    
    if ($actualizados > 0) {
        $mensaje .= "$actualizados registros actualizados (T1). ";
    }
    
    if ($errores > 0) {
        $mensaje .= "$errores errores. ";
    }
    
    if ($total_no_encontrados > 0) {
        $mensaje .= "$total_no_encontrados Cestudiantes no encontrados en concentrado. ";
    }
    
    $mensaje .= "Total procesados: " . count($procesados) . " de " . count($cestudiantes_limpios);
    
    error_log("RESULTADO FINAL: " . $mensaje);
    
    echo json_encode([
        'success' => true,
        'message' => $mensaje,
        'estudiantes_encontrados' => $total_estudiantes,
        'insertados' => $insertados,
        'actualizados' => $actualizados,
        'errores' => $errores,
        'no_encontrados' => $total_no_encontrados,
        'total_recibidos' => count($cestudiantes_limpios),
        'control_por' => 'Cestudiante'
    ]);
    
} catch (Exception $e) {
    error_log("EXCEPCIÓN: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

if (isset($conexion)) {
    $conexion->close();
}
?>