<?php
// pasar_a_representante.php - Enviar datos a la tabla representante

// ACTIVAR errores para debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json; charset=utf-8');

try {
    // CONEXIÓN DESDE ARCHIVO EXTERNO
    require_once 'config_db.php';
    
    $conexion = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    
    if ($conexion->connect_error) {
        throw new Exception('Error conectando a la base de datos: ' . $conexion->connect_error);
    }
    
    $conexion->set_charset(DB_CHARSET);

    // Validar datos
    if (!isset($_POST['ids']) || empty($_POST['ids'])) {
        throw new Exception('No se recibieron IDs');
    }
    
    $ids = $_POST['ids'];
    
    // Limpiar IDs - solo números positivos
    $ids_limpios = array_filter($ids, function($id) {
        return is_numeric($id) && $id > 0;
    });
    
    if (empty($ids_limpios)) {
        throw new Exception('No hay IDs válidos');
    }
    
    // Ordenar IDs para mejor legibilidad
    sort($ids_limpios);
    
    error_log("=== PROCESANDO " . count($ids_limpios) . " IDs PARA REPRESENTANTE ===");
    error_log("IDs: " . implode(', ', $ids_limpios));
    
    // Crear placeholders
    $placeholders = str_repeat('?,', count($ids_limpios) - 1) . '?';
    
    // Consulta para obtener estudiantes desde concentrado
    $query = "SELECT id, Docente, Especialidad, Curso, Paralelo, Asignatura, Trimestre, 
                     Cestudiante, Estudiante, `T2.F.M3.PF` as Calificacion                     
              FROM concentrado WHERE id IN ($placeholders)";
    
    error_log("Query SELECT: " . $query);
    
    $stmt = $conexion->prepare($query);
    if (!$stmt) {
        throw new Exception('Error preparando SELECT: ' . $conexion->error);
    }
    
    $types = str_repeat('i', count($ids_limpios));
    $stmt->bind_param($types, ...$ids_limpios);
    
    if (!$stmt->execute()) {
        throw new Exception('Error ejecutando SELECT: ' . $stmt->error);
    }
    
    $result = $stmt->get_result();
    $total_estudiantes = $result->num_rows;
    error_log("Estudiantes encontrados en concentrado: " . $total_estudiantes);
    
    $insertados = 0;
    $errores = 0;
    $duplicados = 0;
    $procesados = [];
    
    while ($estudiante = $result->fetch_assoc()) {
        $id_actual = $estudiante['id'];
        error_log("Procesando ID: " . $id_actual . " - " . $estudiante['Estudiante']);
        $procesados[] = $id_actual;
        
        // Verificar duplicado en representante
        $check = $conexion->prepare("SELECT id FROM `representante` WHERE id = ?");
        $check->bind_param("i", $id_actual);
        $check->execute();
        
        if ($check->get_result()->num_rows > 0) {
            error_log("ID " . $id_actual . " ya existe en representante - OMITIENDO");
            $duplicados++;
            continue;
        }
        
        // DEBUG: Mostrar datos del estudiante
        error_log("Datos estudiante: " . print_r($estudiante, true));
        
        // Insertar en representante - VERSIÓN SIMPLIFICADA
        $insert = $conexion->prepare("INSERT INTO `representante` (
            Docente, id, Especialidad, Curso, Paralelo, Asignatura, Identificador, 
            Cestudiante, Estudiante, Fecha, Motivo, Compromiso, Trimestre, Calificacion
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, CURDATE(), ?, ?, ?, ?)");
        
        if (!$insert) {
            $error_msg = "Error preparando INSERT para ID " . $id_actual . ": " . $conexion->error;
            error_log($error_msg);
            $errores++;
            continue;
        }
        
        // Valores por defecto
        $motivo_default = "Bajo rendimiento académico";
        $compromiso_default = "Seguimiento académico";
        $identificador = "T2.F.M3.PF";
        
        // DEBUG: Verificar tipos de datos
        error_log("Tipos de datos:");
        error_log("Docente: " . gettype($estudiante['Docente']) . " - " . $estudiante['Docente']);
        error_log("ID: " . gettype($estudiante['id']) . " - " . $estudiante['id']);
        error_log("Calificacion: " . gettype($estudiante['Calificacion']) . " - " . $estudiante['Calificacion']);
        
        // Bind parameters - versión corregida
        $insert->bind_param("sissssssssssd", 
            $estudiante['Docente'],              // s
            $estudiante['id'],                   // i  
            $estudiante['Especialidad'],         // s
            $estudiante['Curso'],                // s
            $estudiante['Paralelo'],             // s
            $estudiante['Asignatura'],           // s
            $identificador,                      // s
            $estudiante['Cestudiante'],          // s
            $estudiante['Estudiante'],           // s
            $motivo_default,                     // s
            $compromiso_default,                 // s
            $estudiante['Trimestre'],            // s
            $estudiante['Calificacion']          // d
        );
        
        if ($insert->execute()) {
            error_log("✓ ID " . $id_actual . " insertado correctamente en representante");
            $insertados++;
        } else {
            $error_msg = "✗ Error insertando ID " . $id_actual . " en representante: " . $insert->error;
            error_log($error_msg);
            $errores++;
        }
        
        $insert->close();
    }
    
    // Verificar si hubo IDs que no se encontraron en concentrado
    $no_encontrados = array_diff($ids_limpios, $procesados);
    if (!empty($no_encontrados)) {
        error_log("IDs no encontrados en concentrado: " . implode(', ', $no_encontrados));
    }
    
    // Respuesta final
    $mensaje = "Proceso completado: $insertados nuevos registros insertados en representante";
    if ($duplicados > 0) $mensaje .= ", $duplicados duplicados omitidos";
    if ($errores > 0) $mensaje .= ", $errores errores de inserción";
    if (!empty($no_encontrados)) $mensaje .= ", " . count($no_encontrados) . " IDs no encontrados en concentrado";
    
    error_log("RESULTADO FINAL: " . $mensaje);
    
    echo json_encode([
        'success' => true,
        'message' => $mensaje,
        'estudiantes_encontrados' => $total_estudiantes,
        'insertados' => $insertados,
        'duplicados' => $duplicados,
        'errores' => $errores,
        'no_encontrados' => count($no_encontrados),
        'total_ids_recibidos' => count($ids_limpios)
    ]);
    
} catch (Exception $e) {
    error_log("EXCEPCIÓN: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

if (isset($conexion)) {
    $conexion->close();
}
?>